<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    die('Not authorized');
}

// Get invoice ID
$invoice_id = intval($_GET['id'] ?? 0);
if ($invoice_id <= 0) {
    die('Invalid invoice ID');
}

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get logo from database
$stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
$stmt->execute();
$logo = $stmt->fetch();
$logoDataUrl = '';
if ($logo && $logo['setting_binary']) {
    $mimeType = 'image/jpeg'; // Default to JPEG
    if (strpos($logo['setting_value'], '.png') !== false) {
        $mimeType = 'image/png';
    } elseif (strpos($logo['setting_value'], '.gif') !== false) {
        $mimeType = 'image/gif';
    }
    $logoDataUrl = 'data:' . $mimeType . ';base64,' . base64_encode($logo['setting_binary']);
}

// Get invoice details
try {
    $stmt = $pdo->prepare("
        SELECT 
            i.*,
            c.name as customer_name,
            c.email as customer_email,
            c.phone as customer_phone,
            c.address as customer_address,
            u.username as created_by_name,
            COALESCE(q.quote_number, '') as quotation_reference,
            COALESCE(SUM(p.amount), 0) as total_paid
        FROM invoices i
        LEFT JOIN customers c ON i.customer_id = c.id
        LEFT JOIN users u ON i.created_by = u.id
        LEFT JOIN quotations q ON i.quotation_id = q.id
        LEFT JOIN payments p ON i.id = p.invoice_id
        WHERE i.id = ?
        GROUP BY i.id
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$invoice) {
        die('Invoice not found');
    }
    
    // Get invoice items
    $stmt = $pdo->prepare("
        SELECT 
            ii.*,
            p.name as product_name
        FROM invoice_items ii
        LEFT JOIN products p ON ii.product_id = p.id
        WHERE ii.invoice_id = ?
        ORDER BY ii.id
    ");
    $stmt->execute([$invoice_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

$totalPaid = floatval($invoice['total_paid'] ?? 0);
$balance = floatval($invoice['total_amount']) - $totalPaid;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice <?php echo htmlspecialchars($invoice['invoice_number']); ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @media print {
            html, body {
                margin: 0 !important;
                padding: 0 !important;
                background: none !important;
            }
            .invoice-container {
                width: 210mm !important;
                min-height: 297mm !important;
                max-width: 210mm !important;
                margin: 0 !important;
                padding: 0 !important;
                box-shadow: none !important;
                border-radius: 0 !important;
                background: white !important;
            }
            .invoice-header {
                background: linear-gradient(135deg, #6f42c1 0%, #e83e8c 100%) !important;
                color: white !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
            .invoice-header h1,
            .invoice-header p,
            .header-info p {
                color: white !important;
            }
            .no-print {
                display: none !important;
            }
        }
        
        body {
            font-family: 'Arial', sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f8f9fa;
            font-size: 12px;
        }
        
        .invoice-container {
            max-width: 210mm;
            width: 210mm;
            min-height: 297mm;
            margin: 0 auto;
            background: white;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            border-radius: 10px;
            overflow: hidden;
            padding: 15mm;
            box-sizing: border-box;
        }
        
        .invoice-header {
            background: linear-gradient(135deg, #6f42c1 0%, #e83e8c 100%);
            color: white;
            padding: 20px;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
            align-items: center;
        }
        
        .header-left {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .store-logo {
            width: 60px;
            height: 60px;
            object-fit: contain;
            border-radius: 8px;
            background: white;
            padding: 5px;
        }
        
        .header-info {
            text-align: right;
        }
        
        .header-info p {
            margin: 2px 0;
            font-size: 11px;
        }
        
        .header-right {
            text-align: right;
        }
        
        .invoice-title-section {
            text-align: center;
        }
        
        .store-info-section {
            text-align: right;
            display: flex;
            flex-direction: column;
            align-items: flex-end;
            gap: 10px;
        }
        
        .invoice-header h1 {
            margin: 0;
            font-size: 24px;
            font-weight: bold;
        }
        
        .invoice-header p {
            margin: 3px 0;
            font-size: 12px;
        }
        
        .invoice-body {
            padding: 20px;
        }
        
        .invoice-info {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .invoice-details, .customer-details {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            border-left: 3px solid #6f42c1;
        }
        
        .invoice-details h3, .customer-details h3 {
            color: #6f42c1;
            border-bottom: 2px solid #6f42c1;
            padding-bottom: 8px;
            margin-bottom: 12px;
            font-size: 14px;
            margin-top: 0;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 6px;
            font-size: 11px;
        }
        
        .detail-label {
            font-weight: bold;
            color: #666;
        }
        
        .detail-value {
            color: #333;
        }
        
        .items-table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
            font-size: 11px;
        }
        
        .items-table th {
            background-color: #6f42c1;
            color: white;
            padding: 8px;
            text-align: left;
            font-size: 11px;
        }
        
        .items-table td {
            padding: 8px;
            border-bottom: 1px solid #ddd;
        }
        
        .items-table tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        
        .totals-section {
            margin-top: 20px;
            text-align: right;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 6px;
            font-size: 12px;
        }
        
        .total-row.grand-total {
            font-size: 14px;
            font-weight: bold;
            color: #6f42c1;
            border-top: 2px solid #6f42c1;
            padding-top: 8px;
        }
        
        .payment-status {
            margin-top: 15px;
            padding: 10px;
            border-radius: 4px;
            text-align: center;
            font-weight: bold;
            font-size: 11px;
        }
        
        .payment-status.paid {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .payment-status.partial {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .payment-status.pending {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .invoice-footer {
            background-color: #f8f9fa;
            padding: 15px 20px;
            text-align: center;
            color: #666;
            border-top: 1px solid #ddd;
            font-size: 10px;
        }
        
        .print-button {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #6f42c1;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
        }
        
        .print-button:hover {
            background: #5a32a3;
        }
        
        .status-badge {
            display: inline-block;
            padding: 3px 10px;
            border-radius: 15px;
            font-size: 10px;
            font-weight: bold;
        }
        
        .status-paid { background-color: #28a745; color: white; }
        .status-pending { background-color: #ffc107; color: #212529; }
        .status-overdue { background-color: #dc3545; color: white; }
    </style>
</head>
<body>
    <button class="print-button no-print" onclick="window.print()">
        <i class="fas fa-print"></i> Print Invoice
    </button>
    
    <div class="invoice-container">
        <div class="invoice-header">
            <div class="invoice-title-section">
                <h1>INVOICE</h1>
            </div>
            <div class="store-info-section">
                <?php if (!empty($logoDataUrl)): ?>
                <img src="<?php echo htmlspecialchars($logoDataUrl); ?>" alt="Store Logo" class="store-logo">
                <?php endif; ?>
                <div class="header-info">
                    <p><?php echo htmlspecialchars($settings['store_name'] ?? 'Your Company Name'); ?></p>
                    <p><?php echo htmlspecialchars($settings['store_address'] ?? 'Company Address'); ?></p>
                    <p>Phone: <?php echo htmlspecialchars($settings['store_phone'] ?? 'Phone Number'); ?></p>
                    <p>Email: <?php echo htmlspecialchars($settings['store_email'] ?? 'email@company.com'); ?></p>
                    <?php if (!empty($settings['website'])): ?>
                    <p>Website: <?php echo htmlspecialchars($settings['website']); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="invoice-body">
            <div class="invoice-info">
                <div class="invoice-details">
                    <h3>Invoice Details</h3>
                    <div class="detail-row">
                        <span class="detail-label">Invoice Number:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['invoice_number']); ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Issue Date:</span>
                        <span class="detail-value"><?php echo date('F j, Y', strtotime($invoice['issue_date'])); ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Due Date:</span>
                        <span class="detail-value"><?php echo date('F j, Y', strtotime($invoice['due_date'])); ?></span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Status:</span>
                        <span class="detail-value">
                            <span class="status-badge status-<?php echo $invoice['status']; ?>">
                                <?php echo ucfirst($invoice['status']); ?>
                            </span>
                        </span>
                    </div>
                    <?php if ($invoice['quotation_reference']): ?>
                    <div class="detail-row">
                        <span class="detail-label">Quotation Ref:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['quotation_reference']); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div class="customer-details">
                    <h3>Customer Information</h3>
                    <div class="detail-row">
                        <span class="detail-label">Customer:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['customer_name'] ?? 'Walk-in Customer'); ?></span>
                    </div>
                    <?php if ($invoice['customer_email']): ?>
                    <div class="detail-row">
                        <span class="detail-label">Email:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['customer_email']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($invoice['customer_phone']): ?>
                    <div class="detail-row">
                        <span class="detail-label">Phone:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['customer_phone']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($invoice['customer_address']): ?>
                    <div class="detail-row">
                        <span class="detail-label">Address:</span>
                        <span class="detail-value"><?php echo htmlspecialchars($invoice['customer_address']); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <table class="items-table">
                <thead>
                    <tr>
                        <th>Item</th>
                        <th>Quantity</th>
                        <th>Unit Price</th>
                        <th>Total</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                        <td><?php echo $item['quantity']; ?></td>
                        <td>Ksh <?php echo number_format($item['unit_price'], 2); ?></td>
                        <td>Ksh <?php echo number_format($item['total_price'], 2); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <div class="totals-section">
                <div class="total-row">
                    <span>Subtotal:</span>
                    <span>Ksh <?php echo number_format($invoice['subtotal'], 2); ?></span>
                </div>
                <div class="total-row">
                    <span>Tax (0%):</span>
                    <span>Ksh <?php echo number_format($invoice['tax_amount'], 2); ?></span>
                </div>
                <div class="total-row grand-total">
                    <span>Total Amount:</span>
                    <span>Ksh <?php echo number_format($invoice['total_amount'], 2); ?></span>
                </div>
                <div class="total-row">
                    <span>Amount Paid:</span>
                    <span>Ksh <?php echo number_format($totalPaid, 2); ?></span>
                </div>
                <div class="total-row">
                    <span>Balance Due:</span>
                    <span>Ksh <?php echo number_format($balance, 2); ?></span>
                </div>
            </div>
            
            <div class="payment-status <?php echo $balance <= 0 ? 'paid' : ($totalPaid > 0 ? 'partial' : 'pending'); ?>">
                <?php if ($balance <= 0): ?>
                    <i class="fas fa-check-circle"></i> Payment Complete
                <?php elseif ($totalPaid > 0): ?>
                    <i class="fas fa-clock"></i> Partial Payment - Balance: Ksh <?php echo number_format($balance, 2); ?>
                <?php else: ?>
                    <i class="fas fa-exclamation-triangle"></i> Payment Pending - Amount Due: Ksh <?php echo number_format($balance, 2); ?>
                <?php endif; ?>
            </div>
            
            <?php if ($invoice['notes']): ?>
            <div style="margin-top: 20px; padding: 15px; background-color: #f8f9fa; border-radius: 4px;">
                <h4 style="color: #6f42c1; margin-bottom: 8px; font-size: 12px;">Notes</h4>
                <p style="margin: 0; color: #666; font-size: 11px;"><?php echo nl2br(htmlspecialchars($invoice['notes'])); ?></p>
            </div>
            <?php endif; ?>
        </div>
        
        <div class="invoice-footer">
            <p><strong>Thank you for your business!</strong></p>
            <p><?php echo htmlspecialchars($settings['store_name'] ?? 'Your Company Name'); ?></p>
            <?php if (!empty($settings['website'])): ?>
            <p><?php echo htmlspecialchars($settings['website']); ?></p>
            <?php endif; ?>
            <p>Generated on <?php echo date('F j, Y \a\t g:i A'); ?> by <?php echo htmlspecialchars($invoice['created_by_name']); ?></p>
        </div>
    </div>
    
    <script>
        // Auto-print when page loads
        window.onload = function() {
            // Small delay to ensure everything is loaded
            setTimeout(function() {
                window.print();
            }, 500);
        };
    </script>
</body>
</html> 